{*

Richard A. DeVenezia, Copyright 2000

Initial Coding:
Sept 2000

Thanks to:
Paul Dunn for answering my silly questions.

Project:
Create an interesting color gradient bitmap.
The bitmap can be saved and optionally the desktop background image
can be changed to it.

Note: This application only seems to create correct .bmp files when running
in 256 or 24-bit color mode.
On my NT box with Matrox G400, running in 16-bit and 32-bit modes did not create
files that could be opened by Microsoft Photo Edit

*}


unit Unit1;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, ExtCtrls, Buttons, OleCtnrs, ColorGrd, ImgList, Math, ComCtrls,
  ToolWin;

type
  TForm1 = class(TForm)

    ColorDialog: TColorDialog;
    grImage: TImage;
    ColorWrap: TCheckBox;

    { acpXXX are controls that display the properties of the active control point,
      that is the last control point clicked (see TControlPoint)
    }

    AddControlPoint: TButton;
    Save: TButton;
    Close: TBitBtn;
    RemoveControlPoint: TBitBtn;    { Drop site }

    ControlPoint0: TBitBtn; { Leftmost control point }
    ControlPoint1: TBitBtn; { Rightmost control point }
    ControlPoint2: TBitBtn; { Interior control points }
    ControlPoint3: TBitBtn;
    ControlPoint4: TBitBtn;
    ControlPoint5: TBitBtn;
    ControlPoint6: TBitBtn;
    ControlPoint7: TBitBtn;
    ControlPoint8: TBitBtn;
    lblRightClick: TLabel;

    lblLocation: TLabel;
    acpAlpha: TEdit;
    acpAlphaWander: TCheckBox;

    lblColor: TLabel;
    acpColor: TEdit;
    acpColorWander: TCheckBox;

    lblWander: TLabel;
    lblInterval: TLabel;
    acpWanderInterval: TEdit;
    lblMs: TLabel;

    rgInterpolation: TRadioGroup;
    acpPoly: TRadioButton;
    acpPolyPower: TComboBox;
    acpSinus: TRadioButton;
    acpSinusShape: TComboBox;

    InterpolationGraph: TGroupBox;
    InterpolationImage: TImage;

    ScaleToScreen: TCheckBox;
    SetWallpaper: TCheckBox;
    VerticalImage: TCheckBox;
    LiveUpdate: TCheckBox;

    SaveDialog1: TSaveDialog;

    procedure frmCreate(Sender: TObject);
    procedure SaveClick(Sender: TObject);
    procedure AlphaWanderClick(Sender: TObject);
    procedure FormDragOver(Sender, Source: TObject; X, Y: Integer;
      State: TDragState; var Accept: Boolean);
    procedure ColorButtonStartDrag(Sender: TObject;
      var DragObject: TDragObject);
    procedure ControlPointEnter(Sender: TObject);
    procedure frmDestroy(Sender: TObject);
    procedure AddControlPointClick(Sender: TObject);
    procedure PickColor(Sender: TObject; MousePos: TPoint;
      var Handled: Boolean);
    procedure ColorButtonEndDrag(Sender, Target: TObject; X, Y: Integer);
    procedure RemoveControlPointDragDrop(Sender, Source: TObject; X,
      Y: Integer);
    procedure ColorWrapClick(Sender: TObject);
    procedure InterpolationFunctionPick(Sender: TObject);
    procedure acpSinusShapeChange(Sender: TObject);
    procedure acpPolyPowerChange(Sender: TObject);
    procedure FormResize(Sender: TObject);
    procedure ColorWanderClick(Sender: TObject);
    procedure acpWanderIntervalExit(Sender: TObject);
    procedure acpWanderIntervalKeyPress(Sender: TObject; var Key: Char);

  private
    { Private declarations }
    { resize parameters }
    ControlPoint1LeftToEdge: integer;
    grImageEdgeGap: integer;
    ColorWrapLeftToEdge: integer;
    SaveLeftToEdge: integer;
    ScaleToScreenLeftToEdge: integer;
    SetWallpaperLeftToEdge: integer;
    VerticalImageLeftToEdge: integer;
    LiveUpdateLeftToEdge: integer;
    CloseLeftToEdge: integer;

    procedure PlotInterpolationGraph;
  public
    { Public declarations }
  end;

var
  Form1: TForm1;

implementation

{$R *.DFM}
{$R DROPPAGE.RES}

const
  N_CONTROL_POINTS = 9;
  crDropPage = 100;             { hand dropping a page cursor }
  inSave: boolean = false;

type
  { Interpolation mapping function
    0 <= f(x) <= 1
    f(0) = 0
    f(1) = 1
  }
  _0to1_MappingFunction = function (arg1:Double; arg2:Double):Double;

  TControlPoint = class (TObject)
    destructor Destroy; override;
  private
    Active: boolean;
    Button: TBitBtn;
    Alpha: Double;  { 0..1, location of control point along dimension }
    Color: TColor;  { color of control point }
    Mapping: _0to1_MappingFunction;
    PolyPower: integer; { PolyPower or SinusShape }
    SinusShape: integer;
    ppIndex: integer; { for combobox }
    ssIndex: integer; { for combobox }
    Timer: TTimer;  { intervals are in ms }
    AlphaWander: Boolean;
    AlphaWanderSign: integer; { -1 or 1 }
    ColorWander: Boolean;
    rSign, gSign, bSign: integer; { -1 or 1 }
    id : integer;
  end;

  TFunkyGradient = class(TObject)
    destructor Destroy; override;
  private
    N : integer;
    ControlPoint : array [0..N_CONTROL_POINTS-1] of TControlPoint;
    ActiveControlPoint : TControlPoint;
    Image: TImage;
    TempFilename: String;
    procedure TimerFired (Sender: TObject);
  end;

var
  DragOffset : integer;
  fg : TFunkyGradient;

destructor TControlPoint.Destroy;
begin
    Timer.Free;
    inherited Destroy;
end;

destructor TFunkyGradient.Destroy;
var
    i : integer;
begin
    for i := 0 to N_CONTROL_POINTS-1 do begin
        fg.ControlPoint[i].Free;
    end;
    inherited Destroy;
end;


function CreateTempFileName(aPrefix: string): string;
var
  Buf: array[0..MAX_PATH] of char;
  Temp: array[0..MAX_PATH] of char;
begin
  GetTempPath(MAX_PATH, Buf);
  if GetTempFilename(Buf, PChar(aPrefix), 0, Temp) = 0 then
    raise Exception.CreateFmt('Win32 Error %d: %s', [GetLastError, SysErrorMessage(GetLastError)]);
  Result := String(Temp);
end;


function PowerMapping (arg1:Double; arg2:Double=0.0): Double;
{ given, arg1 in 0..1, return a value such that
  PowerMapping(0)=0 and
  PowerMapping(1)=1
}
begin
    if arg2 = 0 then arg2 := 1;

    if arg2 < 0
        then PowerMapping := 1 - Power ((1-arg1), -arg2)
        else PowerMapping := Power (arg1, arg2);
end;


function SinusMapping (arg1:Double; arg2:Double=0.0): Double;
{ given, arg1 in 0..1, return a value such that
  PowerMapping(0)=0 and
  PowerMapping(1)=1
}
begin
    if arg2 < 0
        then SinusMapping := Sin (arg1 * Pi / 2.0 + 3.0 * Pi / 2.0) + 1
        else SinusMapping := Sin (arg1 * Pi / 2.0);
end;


procedure SetButtonGlyph(
    Button: TBitBtn;
    WidthInset: Integer;
    HeightInset: Integer);
begin
    Button.Glyph.Width := Button.Width - WidthInset;
    Button.Glyph.Height := Button.Height - HeightInset;
end;


procedure SetButtonGlyphColor(Button: TBitBtn; Color: TColor);
begin
    with Button.Glyph.Canvas do begin
        Pen.Color := clBlack;
        If Color = clWhite
          then Brush.Color := $FFFFFE
          else Brush.Color := Color;

        Brush.Style := bsSolid;

        Rectangle(0, 0, Button.Glyph.Width-1, Button.Glyph.Height-1);
    end;

    Button.Tag := Color;
end;

const
    inSetControlPointColor: boolean = false;
procedure SetControlPointColor (cp: TControlPoint; color: TColor);
begin
    SetButtonGlyphColor(cp.Button, color);
    cp.Color := color;

    if (Form1.ColorWrap.State = cbChecked) and (not inSetControlPointColor) then
    begin
        inSetControlPointColor := true;

        if (cp = fg.ControlPoint[0]) then
            SetControlPointColor (fg.ControlPoint[1], color)
        else
        if (cp = fg.ControlPoint[1]) then
            SetControlPointColor (fg.ControlPoint[0], color);
    end;

    inSetControlPointColor := false;
end;


function AlphaCompare (item1, item2: Pointer) : Integer;
{ for sorting control point locations }
var
    x, y : double;
begin
    x := TControlPoint(item1).Alpha;
    y := TControlPoint(item2).Alpha;

    if x < y then
        AlphaCompare := -1
    else
    if x > y then
        AlphaCompare := 1
    else
        AlphaCompare := 0;
end;


procedure GenerateGradient (fg: TFunkyGradient);
var
    i,j, r,r1,r2,g,g1,g2,b,b1,b2, pxStart, pxEnd: Integer;
    color, color1, color2: TColor;
    factor: Extended;
    cp1, cp2 : TControlPoint;
    L: TList;
    mapArg: double;
begin
    L := TList.Create;
    L.Capacity := N_CONTROL_POINTS;
    for i := 0 to N_CONTROL_POINTS-1 do
        if fg.ControlPoint[i].Active then L.Add (fg.ControlPoint[i]);

    L.Sort(AlphaCompare);

    for i := 1 to L.Count-1 do begin

        cp1 := TControlPoint ( L.Items[i-1]);
        cp2 := TControlPoint ( L.Items[i  ]);

        color1 := cp1.Color;
        color2 := cp2.Color;

        r1 := (ColorToRGB(color1) and $FF0000) shr 16;
        g1 := (ColorToRGB(color1) and $00FF00) shr 8;
        b1 := (ColorToRGB(color1) and $0000FF);

        r2 := (ColorToRGB(color2) and $FF0000) shr 16;
        g2 := (ColorToRGB(color2) and $00FF00) shr 8;
        b2 := (ColorToRGB(color2) and $0000FF);

        with fg.Image.Canvas do begin
            Brush.Style := bsSolid;
            Brush.Color := color1;

            if fg.Image.Width > fg.Image.Height then begin
              pxStart := Trunc (fg.Image.Width * cp1.Alpha);
              pxEnd   := Trunc (fg.Image.Width * cp2.Alpha);
            end
            else begin
              pxStart := Trunc (fg.Image.Height * cp1.Alpha);
              pxEnd   := Trunc (fg.Image.Height * cp2.Alpha);
            end;

            Pen.Color := color1;

            if fg.Image.Width > fg.Image.Height then begin
                MoveTo (pxStart,0);
                LineTo (pxStart,fg.Image.Height);
            end
            else begin
                MoveTo (0,pxStart);
                LineTo (fg.Image.Width,pxStart);
            end;

            if Addr(cp1.Mapping) = Addr(PowerMapping) then
                mapArg := cp1.PolyPower
            else
                mapArg := cp1.SinusShape;

            for j := pxStart+1 to pxEnd-1 do begin

                factor := 1.0 * (j - pxStart) / (pxEnd - pxStart);
                factor := cp1.Mapping (factor, mapArg);

                r := r1 + Trunc ((r2 - r1) * factor);
                g := g1 + Trunc ((g2 - g1) * factor);
                b := b1 + Trunc ((b2 - b1) * factor);

                color :=
                  ((r shl 16) and $FF0000) or
                  ((g shl  8) and $00FF00) or
                    b;

                Pen.Color := color;

                if fg.Image.Width > fg.Image.Height then begin
                    MoveTo (j, 0);
                    LineTo (j, fg.Image.Height);
                end
                else begin
                    MoveTo (0, j);
                    LineTo (fg.Image.Width, j);
                end
            end;

            Pen.Color := color2;
            if fg.Image.Width > fg.Image.Height then begin
                MoveTo (pxEnd,0);
                LineTo (pxEnd,fg.Image.Height);
            end
            else begin
                MoveTo (0,pxEnd);
                LineTo (fg.Image.Width,pxEnd);
            end;
        end;
    end;

    L.Free;

    if inSave then exit;

    if Form1.LiveUpdate.State = cbChecked then
      Form1.SaveClick (Form1);
end;


procedure TForm1.SaveClick(Sender: TObject);
var
    saveImage : TImage;
    vert, full: boolean;
    Filename: String;
begin
    inSave := true;

    Filename := '';

    if LiveUpdate.State = cbChecked then
        Filename := fg.TempFilename
    else
    if SaveDialog1.Execute then
        Filename := SaveDialog1.FileName;

    if (Filename <> '') then begin

        vert := VerticalImage.State = cbChecked;
        full := ScaleToScreen.State = cbChecked;

        saveImage := TImage.Create(Form1);
        with saveImage do begin
            fg.Image := saveImage;

            if (not full) and (not vert) then begin
                Width := grImage.Width;
                Height := 1;
            end
            else
            if (full) and (not vert) then begin
                Width := Screen.Width;
                Height := 1;
            end
            else
            if (not full) and (vert) then begin
                Width := 1;
                Height := grImage.Width;
            end
            else
            if (full) and (vert) then begin
                Width := 1;
                Height := Screen.Height;
            end;

            GenerateGradient (fg);
            Picture.SaveToFile (Filename);
            Free;

            fg.Image := grImage;
        end;

        if SetWallpaper.State = cbChecked then
          SystemParametersInfo (
            SPI_SETDESKWALLPAPER
          , 0
          , PChar(Filename)
          , 0); // SPIF_UPDATEINIFILE or SPIF_SENDCHANGE
    end;

    inSave := false;
end;


procedure TForm1.AlphaWanderClick(Sender: TObject);
begin
    If fg.ActiveControlPoint = nil then exit;

    fg.ActiveControlPoint.AlphaWander := boolean (acpAlphaWander.State);

    fg.ActiveControlPoint.Timer.Enabled
    := fg.ActiveControlPoint.AlphaWander
    or fg.ActiveControlPoint.ColorWander;
end;


procedure TForm1.ColorWanderClick(Sender: TObject);
begin
    If fg.ActiveControlPoint = nil then exit;

    fg.ActiveControlPoint.ColorWander := boolean (acpColorWander.State);

    fg.ActiveControlPoint.Timer.Enabled
    := fg.ActiveControlPoint.AlphaWander
    or fg.ActiveControlPoint.ColorWander;
end;


procedure TForm1.FormDragOver(Sender, Source: TObject; X, Y: Integer;
  State: TDragState; var Accept: Boolean);
begin
  { Drag and Drop is used to allow movement of TBitBtn's
    When a TBitBtn is being dragged, and the mouse goes over a component that
    does NOT have this method for it's OnDragOver method, the TBitBtn will not
    be 'moved' until the mouse moves over a component that has it's OnDragOver
    set to this method.

    This method is the onDragOver for every object in the Form.  That will allow
    a TBitBtn to be dragged no matter where the mouse goes.  This method restricts
    the motion to horizontal only, so the button will not be travelling all over
    the form.

    Because the moveable TBitBtn's are DragMode dmAutomatic, when they are clicked
    these events will occur:
    1. TBitBtn will get a OnEnter
    2. The component under the mouse will get a OnDragOver

    When a TBitBtn that is allowed to be dragged is entered (obtains focus),
    the global variable DragOffset is set to 32768.  When dragging starts, this
    method is entered, and the 'just got focus' state is determined by the value
    of DragOffset.  Since we just got focus, the DragOver is sent to the same
    TBitBtn, and X is the button local reference.  DragOffset is recorded as
    -X (for use in later computations) to prevent 'jerky' motion of the button
    as it tracks the mouse.
  }

  Accept := Source is TBitBtn;
  If not Accept then exit;

  { button just got focus, record the x offset within the button }
  If DragOffset = 32768 then begin
    DragOffset := -X;
    exit;
  end;

  { special case for non-form components, make sure X represents mouse
    position with respect to form, not with respect to component }
  if Sender <> Form1 then
      X := X + TControl(Sender).Left;

  If Sender = RemoveControlPoint then
    TBitBtn(Source).DragCursor := crDropPage
  else
    TBitBtn(Source).DragCursor := crHSplit;

  TBitBtn(Source).Left := X + DragOffset;
  RemoveControlPoint.Left := X + DragOffset - RemoveControlPoint.Width div 4;

  fg.ActiveControlPoint.Alpha
  := 1.0 * (TBitBtn(Source).Left + TBitBtn(Source).Width div 2 - fg.Image.Left)
            / fg.Image.Width;

  acpAlpha.Text := Format('%f', [fg.ActiveControlPoint.Alpha]);

  GenerateGradient (fg);
end;


procedure TForm1.ColorButtonStartDrag(Sender: TObject;
  var DragObject: TDragObject);
begin
  DragOffset := 32768;
  RemoveControlPoint.Visible := True;
  RemoveControlPoint.Left := TBitBtn(Sender).Left - RemoveControlPoint.Width div 4;
end;


procedure TForm1.PlotInterpolationGraph;
var
    factor : double;
    mapping : _0to1_MappingFunction;
    mapArg : double;
    X, Y: integer;
    cp : TControlPoint;
begin
    with InterpolationImage.Canvas do begin
        Pen.Color := clBtnFace;
        Brush.Style := bsSolid;
        Brush.Color := clBtnFace;
        Rectangle (0,0,InterpolationImage.Width,InterpolationImage.Height);

        Pen.Color := clBlack;
        MoveTo (0, InterpolationImage.Height);

        cp := fg.ActiveControlPoint;
        mapping := cp.Mapping;
        if Addr(cp.Mapping) = Addr(PowerMapping)
            then mapArg := cp.PolyPower
            else mapArg := cp.SinusShape;

        for X := 1 to InterpolationImage.Width do begin
            factor := 1.0 * X / InterpolationImage.Width;
            factor := mapping (factor, mapArg) / factor;
            Y := InterpolationImage.Height - 1 - Trunc (X * factor) ;
            LineTo (X, Y);
        end;
    end;
{
//  this didna work
    with TCanvas.Create do
    try
        Handle := InterpolationGraph.Handle;
        Pen.Color := clBlack;
        Brush.Style := BsSolid;
        Brush.Color := clBlack;
        MoveTo (0,0);
        LineTo (10,10);
    finally
        Free;
    end;
}
end;


procedure TForm1.ControlPointEnter(Sender: TObject);
var
    ix : integer;
    cp : TControlPoint;
begin
    { Control points are named ControlPoint0 to ControlPoint9 }
    { Hence, the index number text starts at position 13 }
    with Sender as TBitBtn do begin
      ix := StrToInt (Copy(Name,13,2));
      cp := fg.ControlPoint[ix];
      fg.ActiveControlPoint := cp;
      acpAlpha.Text := Format('%f', [cp.Alpha]);
      acpColor.Text := Format('%.6x', [cp.Button.Tag]);
      acpAlphaWander.State := TCheckBoxState ( cp.AlphaWander );
      acpColorWander.State := TCheckBoxState ( cp.ColorWander );
      acpWanderInterval.text := Format ('%d', [cp.Timer.Interval]);
      acpPolyPower.ItemIndex := cp.ppIndex;
      cp.PolyPower := StrToInt (acpPolyPower.Text);
      acpSinusShape.ItemIndex := cp.ssIndex;
      cp.SinusShape := StrToInt (acpSinusShape.Text);

      if Addr (cp.Mapping) = Addr (PowerMapping) then
        acpPoly.Checked := true
      else
        acpSinus.Checked := true;
    end;

    acpAlphaWander.Enabled := NOT ((Sender = ControlPoint0) or (Sender = ControlPoint1));

    PlotInterpolationGraph;
end;


procedure TForm1.AddControlPointClick(Sender: TObject);
{ Add a control point at the point the two adjacent control
  points that are the farthest apart

  Note: This method disables the AddControlPoint button
  when the last control point is added.  The AddControlPoint
  button is re-enabled in the RemoveControlPointDragDrop
  method.
}
var
    cp, cp1, cp2 : TControlPoint;
    BitBtn : TBitBtn;
    L: TList;
    i: integer;
    gap, maxGap, gapCenter: Double;
    X: integer;

begin
    L := TList.Create;
    L.Capacity := N_CONTROL_POINTS;
    for i := 0 to N_CONTROL_POINTS-1 do
        if fg.ControlPoint[i].Active then L.Add (fg.ControlPoint[i]);

    L.Sort(AlphaCompare);

    maxGap := -1;
    gapCenter := 0.5; { just to please the compiler }
    for i := 0 to L.Count-2 do begin

        cp1 := TControlPoint ( L.Items[i  ]);
        cp2 := TControlPoint ( L.Items[i+1]);

        gap := cp2.Alpha - cp1.Alpha;

        if gap > maxGap then begin
            maxGap := gap;
            gapCenter := ( cp1.Alpha + cp2.Alpha ) / 2.0;
        end;
    end;

    if fg.N < N_CONTROL_POINTS then begin
        { find an inactive control point }
        i := -1;
        repeat
            Inc (i);
            cp := fg.ControlPoint[i];
        until not cp.Active;

        cp.Active := true;
        cp.Alpha := gapCenter;
        X := fg.Image.Left + trunc(cp.Alpha * fg.Image.Width);

        BitBtn := cp.Button;
        BitBtn.Visible := True;
        BitBtn.Top := fg.ControlPoint[0].Button.Top;
        BitBtn.Left := X - BitBtn.Width div 2;
        SetControlPointColor(cp, fg.Image.Canvas.Pixels[X,0]);

        Inc (fg.N);
        if fg.N = N_CONTROL_POINTS then
            AddControlPoint.Enabled := false;

        GenerateGradient (fg);
    end;
end;


procedure TForm1.PickColor(Sender: TObject; MousePos: TPoint;
  var Handled: Boolean);
var
    ix : integer;
    cp : TControlPoint;
begin
    ColorDialog.Color := TBitBtn(Sender).Tag;
    If ColorDialog.Execute then begin
      with Sender as TBitBtn do begin
        ix := StrToInt (Copy(Name,13,2));
        cp := fg.ControlPoint[ix];
        SetControlPointColor (cp, ColorDialog.Color);
        acpColor.Text := Format('%.6x', [cp.Button.Tag]);
        GenerateGradient (fg);
      end;
    end;
end;

procedure TForm1.ColorButtonEndDrag(Sender, Target: TObject; X,
  Y: Integer);
begin
  RemoveControlPoint.visible := False;
  DragOffset := 32767;
End;


procedure TForm1.RemoveControlPointDragDrop(Sender, Source: TObject; X,
  Y: Integer);
var
  i : integer;
  cp : TControlPoint;
begin
  i := -1;

  repeat
    Inc (i);
    if fg.ActiveControlPoint = fg.ControlPoint[i] then begin
      cp := fg.ControlPoint[i];
      cp.Button.Visible := false;
      cp.Active := false;
      Dec (fg.N);

      AddControlPoint.Enabled := true;

      GenerateGradient (fg);

      i := N_CONTROL_POINTS;
    end;
  until (i >= N_CONTROL_POINTS-1);

end;


procedure TFunkyGradient.TimerFired (Sender: TObject);
var
    ix : integer; { control point index }
    cp : TControlPoint;
    r, g, b, rDelta, gDelta, bDelta : integer;
    alphaDelta : double;
const
    maxColorWander = 6;
    maxAlphaWander = 2;
begin
    ix := TTimer(Sender).Tag;
    cp := ControlPoint[ix];

    if (cp.AlphaWander) and (ix > 1) and (dragOffset = 32767) then begin
        { 1 in 8 chance of changing 'direction' }
        if Random (8) = 0 then
            cp.AlphaWanderSign := -cp.AlphaWanderSign ;

        alphaDelta := cp.AlphaWanderSign * Random (maxAlphaWander) / 100;
        cp.Alpha := cp.Alpha + alphaDelta;

        if (cp.Alpha < 0) or (cp.Alpha > 1) then
            cp.AlphaWanderSign := -cp.AlphaWanderSign ;

        if cp.Alpha < 0 then
            cp.Alpha := -cp.Alpha
        else
        if cp.Alpha > 1 then
            cp.Alpha := 2 - cp.Alpha;

        cp.Button.Left := Form1.grImage.Left
                        + Trunc (cp.Alpha * Form1.grImage.Width)
                        - cp.Button.Width div 2;

        Form1.acpAlpha.Text := Format('%f', [fg.ActiveControlPoint.Alpha]);
    end;

    if cp.ColorWander then begin
        r := (ColorToRGB(cp.Color) and $FF0000) shr 16;
        g := (ColorToRGB(cp.Color) and $00FF00) shr 8;
        b := (ColorToRGB(cp.Color) and $0000FF);

        { 1 in 20 chance of changing 'direction' }
        if Random (20) = 0 then begin
            cp.rSign := -cp.rSign;
            cp.gSign := -cp.gSign;
            cp.bSign := -cp.bSign;
        end;

        rDelta := Random (maxColorWander) + cp.rSign * maxColorWander;
        gDelta := Random (maxColorWander) + cp.gSign * maxColorWander;
        bDelta := Random (maxColorWander) + cp.bSign * maxColorWander;

        { 1 in 3 chance of getting a spiked delta }
        if Random (3) = 0 then rDelta := rDelta * Random (Random (maxColorWander));
        if Random (3) = 0 then gDelta := gDelta * Random (Random (maxColorWander));
        if Random (3) = 0 then bDelta := bDelta * Random (Random (maxColorWander));

        Inc (r, rdelta);
        Inc (g, gDelta);
        Inc (b, bDelta);

        if (r > 255) or (r < 0) then cp.rSign := -cp.rSign;
        if (g > 255) or (g < 0) then cp.gSign := -cp.gSign;
        if (b > 255) or (b < 0) then cp.bSign := -cp.bSign;

        if r > 255 then r := 255 - (r - 255) else if r < 0 then r := -r;
        if g > 255 then g := 255 - (g - 255) else if g < 0 then g := -g;
        if b > 255 then b := 255 - (b - 255) else if b < 0 then b := -b;

        cp.Color :=
            ((r shl 16) and $FF0000) or
            ((g shl  8) and $00FF00) or
              b;

        SetControlPointColor (cp, cp.Color);
        Form1.acpColor.Text := Format('%.6x', [cp.Color]);
    end;

    GenerateGradient (fg);
end;

procedure TForm1.frmCreate(Sender: TObject);
var
  i : integer;
  BitBtn : TBitBtn;
  Timer : TTimer;
  cp, cp0, cp1 : TControlPoint;
begin
  Screen.Cursors[crDropPage] := LoadCursor(HInstance, 'Cursor_1');

  fg := TFunkyGradient.Create;

  fg.N := 2;
  fg.Image := grImage;

  for i := 0 to N_CONTROL_POINTS-1 do begin
    cp := TControlPoint.Create;
    cp.id := i;
    fg.ControlPoint[i] := cp;

    BitBtn := FindComponent ('ControlPoint'+Format('%d',[i])) as TBitBtn;
    BitBtn.DoubleBuffered := True;
    SetButtonGlyph(BitBtn, 10,10);
    cp.Button := BitBtn;

    Timer := TTimer.Create (TComponent(cp.Button));
    Timer.Enabled := false;
    Timer.Interval := 1000;
    Timer.OnTimer := fg.TimerFired;
    Timer.Tag := i;
    cp.Timer := Timer;

    cp.AlphaWanderSign := 1;
    cp.rSign := 1;
    cp.gSign := 1;
    cp.bSign := 1;

    cp.Mapping := PowerMapping;
    cp.ppIndex := 4;
    cp.PolyPower := StrToInt (acpPolyPower.Items.Strings[cp.ppIndex]);
    cp.ssIndex := 0;
    cp.SinusShape := StrToInt (acpSinusShape.Items.Strings[cp.ssIndex]);

    if i > 1 then begin
      BitBtn.visible := false;
      cp.Active := false;
    end;
  end;

  fg.TempFilename := CreateTempFileName ('gradient');

  cp0 := fg.ControlPoint[0];
  cp1 := fg.ControlPoint[1];

  cp0.Active := true;
  cp1.Active := true;

  cp0.Alpha := 0.0;
  cp1.Alpha := 1.0;

  cp0.Button.Left := grImage.Left - cp0.Button.Width div 2;
  cp1.Button.Left := grImage.Left + grImage.Width - cp1.Button.Width div 2;

  SetControlPointColor (cp0, clBlue);
  SetControlPointColor (cp1, clWhite);

  GenerateGradient (fg);

  ControlPoint1LeftToEdge:= Form1.Width - ControlPoint1.Left;
  ColorWrapLeftToEdge    := Form1.Width - ColorWrap.Left;
  SaveLeftToEdge         := Form1.Width - Save.Left;
  ScaleToScreenLeftToEdge:= Form1.Width - ScaleToScreen.Left;
  SetWallpaperLeftToEdge := Form1.Width - SetWallpaper.Left;
  VerticalImageLeftToEdge:= Form1.Width - VerticalImage.Left;
  CloseLeftToEdge        := Form1.Width - Close.Left;
  LiveUpdateLeftToEdge   := Form1.Width - LiveUpdate.Left;
  grImageEdgeGap         := Form1.Width - (grImage.Left + grImage.Width);

  DragOffset := 32767;
end;


procedure TForm1.frmDestroy(Sender: TObject);
begin
    DeleteFile (PChar(fg.TempFilename));
    fg.Destroy;
end;


procedure TForm1.ColorWrapClick(Sender: TObject);
begin
    If (ColorWrap.State = cbChecked) then begin
        if (fg.N = 2) then begin
            AddControlPointClick(AddControlPoint);
            SetControlPointColor (fg.ControlPoint[2], fg.ControlPoint[1].Color);
            SetControlPointColor (fg.ControlPoint[1], fg.ControlPoint[0].Color);
        end
        else begin
            SetControlPointColor (fg.ControlPoint[1], fg.ControlPoint[0].Color);
        end;

        GenerateGradient (fg);
    end;
end;

procedure TForm1.InterpolationFunctionPick(Sender: TObject);
begin
    if fg.ActiveControlPoint = nil then Exit;

    if Sender = acpPoly then
        fg.ActiveControlPoint.Mapping := PowerMapping
    else
        fg.ActiveControlPoint.Mapping := SinusMapping;

    PlotInterpolationGraph;
    GenerateGradient (fg);
end;

procedure TForm1.acpSinusShapeChange(Sender: TObject);
begin
    if fg.ActiveControlPoint = nil then Exit;

    fg.ActiveControlPoint.ssIndex := TComboBox(Sender).ItemIndex;
    fg.ActiveControlPoint.SinusShape := StrToInt (TComboBox(Sender).Text);

    PlotInterpolationGraph;
    GenerateGradient (fg);
end;

procedure TForm1.acpPolyPowerChange(Sender: TObject);
begin
    if fg.ActiveControlPoint = nil then Exit;

    fg.ActiveControlPoint.ppIndex := TComboBox(Sender).ItemIndex;
    fg.ActiveControlPoint.PolyPower := StrToInt (TComboBox(Sender).Text);

    PlotInterpolationGraph;
    GenerateGradient (fg);
end;

procedure TForm1.FormResize(Sender: TObject);
var
    i : integer;
begin
    { relocate components 'stuck' to right edge }
    ControlPoint1.Left := Form1.Width - ControlPoint1LeftToEdge;
    ColorWrap.Left     := Form1.Width - ColorWrapLeftToEdge;
    Save.Left          := Form1.Width - SaveLeftToEdge;
    ScaleToScreen.Left := Form1.Width - ScaleToScreenLeftToEdge;
    SetWallpaper.Left  := Form1.Width - SetWallpaperLeftToEdge;
    VerticalImage.Left := Form1.Width - VerticalImageLeftToEdge;
    LiveUpdate.Left    := Form1.Width - LiveUpdateLeftToEdge;
    Close.Left         := Form1.Width - CloseLeftToEdge;

    { adjust the image to the new form size }
    grImage.Width      := Form1.Width - grImage.Left - grImageEdgeGap ;
    grImage.Picture.Bitmap.Width := grImage.Width ;

    { relocate the control points along the image }
    for i := 0 to N_CONTROL_POINTS-1 do begin
        if fg.ControlPoint[i].Active then
            fg.ControlPoint[i].Button.Left
            := fg.Image.Left + trunc(fg.ControlPoint[i].Alpha * fg.Image.Width)
             - fg.ControlPoint[i].Button.Width div 2;
    end;

    GenerateGradient (fg);
end;


procedure TForm1.acpWanderIntervalExit(Sender: TObject);
begin
    if fg.ActiveControlPoint = nil then exit;

    if (acpWanderInterval.Text = '') then
        acpWanderInterval.Text := IntToStr (fg.ActiveControlPoint.Timer.Interval)
    else
        fg.ActiveControlPoint.Timer.Interval := StrToInt (acpWanderInterval.Text);
end;


procedure TForm1.acpWanderIntervalKeyPress(Sender: TObject; var Key: Char);
begin
    if fg.ActiveControlPoint = nil then exit;

    if not (key in ['0'..'9', Chr($0D), Chr($08)]) then begin
        Key := Chr ($0); // null
        exit;
    end;

    if key = Chr($0D) then
        if (acpWanderInterval.Text = '') then
            acpWanderInterval.Text := IntToStr (fg.ActiveControlPoint.Timer.Interval)
        else
            fg.ActiveControlPoint.Timer.Interval := StrToInt (acpWanderInterval.Text);
end;


end.

